---
name: writer-friendly-skill-builder
description: Interview-style SKILL builder for non-technical writers. Asks simple questions about desired SKILL functionality, validates all inputs, translates answers into proper skill-creator format, builds SKILLs incrementally to prevent token limits, and packages with validated YAML to prevent upload errors. Use when writers want to create custom writing SKILLs without technical knowledge.
---

# Writer-Friendly SKILL Builder Wrapper

This meta-SKILL makes it easy for non-technical writers to build custom writing SKILLs. It handles all technical complexity, prevents YAML errors, and builds incrementally to avoid token limits.

## How This Works

Writers answer simple questions. This wrapper:
1. Validates all answers
2. Translates to proper skill-creator format
3. Builds incrementally (prevents token limits)
4. Validates YAML before packaging
5. Delivers error-free .zip files ready for upload

## Module 1: Discovery Interview

This module asks writer-friendly questions and translates answers into proper skill-creator specifications.

### Interview Process

Ask questions one at a time. Wait for answers. Validate before continuing.

#### Question 1: Main Purpose

**Ask:**
"What should this SKILL do? In simple terms, what do you want help with?"

**Validation:**
- Must be clear and specific
- If vague (like "help with writing"), ask: "What specific type of writing? Scenes? Outlines? Dialogue?"
- If too broad (like "write novels"), narrow down: "Should this focus on a specific part of novel writing, like chapters, scenes, or plot development?"

**Examples of good answers:**
- "Expand brief scene descriptions into full scenes with dialogue"
- "Generate romance tropes and plot premises"
- "Edit chapters for deep POV and style consistency"
- "Create chapter-by-chapter outlines from a story premise"

#### Question 2: Input Type

**Ask:**
"What will you give this SKILL as input?"

**Options to present:**
- "A few sentences or brief description"
- "A full chapter or scene"
- "An outline or list"
- "Character/worldbuilding details"
- "A manuscript section to edit"
- "Something else (describe it)"

**Validation:**
- Must be concrete
- If unclear, ask: "About how much text? A paragraph? A page? Multiple pages?"

#### Question 3: Output Format

**Ask:**
"What should the output look like?"

**Guide them with options:**
- "How long? (word count range)"
- "What format? (prose, outline, list, dialogue-heavy, etc.)"
- "Any specific structure? (chapters, scenes, sections)"

**Validation:**
- Need at least length/format
- If missing details, prompt: "Should this be in any particular POV or tense?"

#### Question 4: Rules & Constraints

**Ask:**
"Any specific rules or preferences? For example:
- POV requirements (third person, first person, deep POV)
- Tense (past, present)
- Heat level (if romance)
- Words/phrases you never use
- Style preferences"

**Validation:**
- This can be "none" or "no specific rules"
- If they mention forbidden elements, capture exactly
- If they mention style, capture specifics

**Common answers:**
- "Never use em dashes or ellipses"
- "Always write in third person limited, past tense"
- "Keep it at medium heat level"
- "Deep POV only, no head-hopping"
- "British spelling"

#### Question 5: Workflow Complexity

**Ask:**
"Does this need multiple steps, or is it one task?"

**Guide them:**
- "One task example: Take input, write output"
- "Multiple steps example: Plan outline, then write scenes, then edit"

**Validation:**
- Determines if single-module or multi-module SKILL
- If multi-step, ask: "What are the steps in order?"

#### Question 6: Skill Name

**Ask:**
"What should we name this SKILL? Keep it simple and descriptive."

**Automatic formatting:**
- Convert to lowercase
- Replace spaces with hyphens
- Remove special characters
- Remove underscores
- Validate: only lowercase letters, numbers, hyphens

**Examples:**
- User says: "Scene Expander" → Format as: `scene-expander`
- User says: "Romance Plot Generator!" → Format as: `romance-plot-generator`
- User says: "Deep_POV_Editor" → Format as: `deep-pov-editor`

**If name is too long (>50 characters):**
"That name is a bit long. How about shortening it to: [suggested shorter version]?"

### Generating the Specification

After all questions are answered and validated, generate a complete specification document in this format:

```
# SKILL Specification: [skill-name]

## Purpose
[What the SKILL does - from Question 1]

## Input
[What user provides - from Question 2]

## Output
[What SKILL produces - from Question 3]

## Rules & Constraints
[Specific requirements - from Question 4, or "None specified"]

## Workflow Type
[Single-module or Multi-module - from Question 5]

## Workflow Steps (if multi-module)
1. [Step 1]
2. [Step 2]
3. [Step 3]

## Skill Name
[formatted-skill-name]

## For skill-creator
This SKILL should:
- [Translate Purpose into skill-creator task description]
- Accept input: [Input type]
- Produce output: [Output format and length]
- Enforce rules: [List all rules/constraints]
- [If multi-module] Use modular workflow with sections: [list modules]

## Validation Checklist
✓ Skill name is lowercase-with-hyphens-only
✓ Purpose is clear and specific
✓ Input type is concrete
✓ Output format is defined
✓ Rules are captured (or marked as "none")
✓ Workflow type is determined
```

### Name Validation Function

Before finalizing the specification, validate the skill name:

**Valid characters:**
- Lowercase letters (a-z)
- Numbers (0-9)
- Hyphens (-)

**Invalid patterns:**
- Uppercase letters
- Spaces
- Underscores
- Special characters (!@#$%^&*()+=[]{}|;:'",.<>?/\)
- Starting or ending with hyphen
- Double hyphens (--)

**Auto-fix process:**
1. Convert to lowercase
2. Replace spaces with single hyphen
3. Replace underscores with single hyphen
4. Remove all special characters
5. Replace multiple consecutive hyphens with single hyphen
6. Remove leading/trailing hyphens
7. Validate result contains only: a-z, 0-9, -

**If name fails validation after auto-fix:**
Tell user: "The name '[original]' needs adjustment. I've formatted it as '[fixed-version]'. Is that okay, or would you prefer a different name?"

### Confirmation Step

After generating the specification, show it to the writer and ask:
"Does this look right? Anything you'd like to change?"

If changes needed, update the spec. If approved, proceed to tell them:
"Perfect! Your specification is ready. This SKILL will be built incrementally to avoid token limits. Module 2 will handle the actual building process."

### What Module 1 Delivers

Module 1 produces:
1. **Validated specification document** - Ready for skill-creator
2. **Properly formatted skill name** - No upload errors
3. **Clear input/output definitions** - No ambiguity
4. **Complete rules list** - All constraints captured
5. **Workflow structure** - Single or multi-module determined

This specification becomes the input for Module 2 (Incremental Builder).

---

## Module 1 Complete

Module 1 (Discovery Interview) is now complete. This module handles:
- ✓ Writer-friendly question flow
- ✓ Answer validation at each step
- ✓ Automatic name formatting
- ✓ Specification generation
- ✓ Ready for Module 2 handoff

**Next:** Module 2 will use this specification to build the SKILL incrementally with skill-creator, preventing token limit issues.

---

## Module 2: Incremental Builder

This module works with skill-creator to build SKILLs in small sections, saving after each section to prevent token limit issues. It provides auto-save functionality, progress tracking, resume capability, and token monitoring.

### Purpose

Build complete SKILLs incrementally without hitting token limits. Save work at each checkpoint. Allow resuming if interrupted.

### How It Works

Module 2 takes the validated specification from Module 1 and coordinates with skill-creator to build the SKILL content section by section, ensuring no work is lost and token limits are never exceeded.

**Process Flow:**
```
Specification from Module 1
    ↓
Analyze workflow type (single or multi-module)
    ↓
Build first section with skill-creator
    ↓
SAVE section to file
    ↓
Update progress tracker
    ↓
Check token usage
    ↓
Continue to next section (or finish if complete)
    ↓
Assemble all saved sections
    ↓
Deliver complete SKILL content to Module 3
```

### Section-by-Section Building Strategy

#### For Single-Module SKILLs

Single-module SKILLs perform one task: input → process → output.

**Building steps:**
1. Read specification from Module 1
2. Tell user: "This is a single-module SKILL. Building now..."
3. Instruct skill-creator: "Build a SKILL with these specifications: [full spec]"
4. Save the complete SKILL body to `/home/claude/writer-friendly-skill-builder/sections/main-module.txt`
5. Update progress: "Main module complete (1/1)"
6. Confirm: "Single-module SKILL build complete!"

#### For Multi-Module SKILLs

Multi-module SKILLs have distinct workflow steps (plan → write → edit, etc.).

**Building steps:**
1. Read specification from Module 1
2. Identify modules from "Workflow Steps" section
3. Tell user: "This is a multi-module SKILL with [N] modules. Building incrementally..."
4. For each module:
   - Tell user: "Building Module [X] of [N]: [module name]"
   - Instruct skill-creator: "Build only Module [X] with these requirements: [module spec]"
   - Save to `/home/claude/writer-friendly-skill-builder/sections/module-[X].txt`
   - Update progress tracker
   - Check token usage
   - Tell user: "Module [X] complete and saved"
5. After all modules built, assemble complete SKILL
6. Confirm: "All [N] modules complete!"

### Auto-Save Functionality

Save after completing each section to prevent loss of work.

**Save process:**
```python
def save_section(section_name, content):
    """Save completed section to file"""
    filepath = f"/home/claude/writer-friendly-skill-builder/sections/{section_name}.txt"
    
    # Write content to file
    write_file(filepath, content)
    
    # Update progress tracker
    update_progress(section_name, "complete")
    
    # Confirm to user
    print(f"✓ {section_name} saved to {filepath}")
    
    return filepath
```

**What gets saved:**
- Module content (complete text)
- Section name (for tracking)
- Timestamp (when saved)
- Token count at save point

**Save locations:**
- Individual sections: `/home/claude/writer-friendly-skill-builder/sections/`
- Progress tracker: `/home/claude/writer-friendly-skill-builder/build-progress.txt`
- Assembly file: `/home/claude/writer-friendly-skill-builder/assembled-skill.md`

### Progress Tracking System

Track what's been completed, what's in progress, and what remains.

**Progress file format:**
```
SKILL Name: [skill-name]
Total Modules: [N]
Status: [In Progress / Complete]

Module Status:
[ ] Module 1: [name] - Not started
[✓] Module 2: [name] - Complete (saved to module-2.txt)
[→] Module 3: [name] - In progress
[ ] Module 4: [name] - Not started

Token Usage: [current] / 190000
Last Updated: [timestamp]
```

**Update progress after each action:**
```python
def update_progress(module_name, status):
    """Update progress tracking file"""
    progress_file = "/home/claude/writer-friendly-skill-builder/build-progress.txt"
    
    # Read current progress
    current = read_file(progress_file)
    
    # Update status for this module
    updated = mark_module_status(current, module_name, status)
    
    # Write back to file
    write_file(progress_file, updated)
```

**Display progress to user:**
- After each save: "✓ Module 2 of 5 complete"
- During work: "→ Building Module 3 of 5..."
- At completion: "✓ All 5 modules complete!"

### Resume Capability

If Claude stops mid-build (token limit, interruption, etc.), resume from last saved checkpoint.

**Resume detection:**
When Module 2 starts, check if build-progress.txt exists:
- If yes: Read it and determine what's complete
- If no: This is a fresh build

**Resume process:**
```python
def resume_build():
    """Resume building from last checkpoint"""
    progress_file = "/home/claude/writer-friendly-skill-builder/build-progress.txt"
    
    if file_exists(progress_file):
        # Read progress
        progress = read_file(progress_file)
        
        # Parse what's complete
        completed_modules = get_completed_modules(progress)
        next_module = get_next_module(progress)
        
        # Tell user
        print(f"Resuming build. Completed: {completed_modules}")
        print(f"Now building: {next_module}")
        
        # Continue from next module
        return next_module
    else:
        # Fresh build
        return "start_from_beginning"
```

**User messaging for resume:**
"I see we were building '[skill-name]' and completed [list of modules]. Resuming now with [next module]..."

### Token Monitoring

Check token usage periodically and warn before hitting limits.

**Monitoring points:**
- After each section save
- Before starting new section
- When assembling final SKILL

**Token thresholds:**
```
< 50,000 tokens used: Continue normally
50,000-100,000: Monitor closely
100,000-140,000: Warn user, prepare to save
140,000-160,000: SAVE IMMEDIATELY, suggest new chat
> 160,000: STOP, save, must continue in new chat
```

**Warning messages:**

At 100k tokens:
"Token usage: 100,000 / 190,000. We have plenty of room. Continuing..."

At 140k tokens:
"Token usage: 140,000 / 190,000. Approaching limits. Saving progress now. We may need to continue in a new chat for remaining modules."

At 160k tokens:
"Token usage: 160,000 / 190,000. SAVING IMMEDIATELY. 

To continue:
1. Download the current progress files
2. Start a new chat
3. Upload the progress files
4. Say: 'Resume building [skill-name]'

All completed work is saved and will not be lost."

### Coordinating with skill-creator

Module 2 instructs skill-creator on what to build. It does NOT build the SKILL content itself.

**How to instruct skill-creator:**

For single-module SKILLs:
```
I need skill-creator to build a SKILL with these specifications:

[Insert complete specification from Module 1]

Build the complete SKILL body with:
- Clear instructions for the task
- Input/output examples
- All rules and constraints enforced
- Proper formatting and structure

Focus only on the SKILL content. I will handle YAML frontmatter separately.
```

For multi-module SKILLs:
```
I need skill-creator to build Module [X] of [N] for a SKILL.

Module [X] specifications:
Purpose: [module purpose]
Input: [what this module receives]
Output: [what this module produces]
Rules: [module-specific rules]

Build only this module. I will call you again for the next module.

Focus only on this module's content. I will handle assembly and YAML separately.
```

**Important:** Module 2 coordinates but doesn't write SKILL content. It tells skill-creator what to write, saves the results, tracks progress, and monitors tokens.

### Assembling Complete SKILL

After all sections are saved, assemble them into one complete SKILL file.

**Assembly process:**
```python
def assemble_skill(skill_name):
    """Combine all saved sections into complete SKILL"""
    sections_dir = "/home/claude/writer-friendly-skill-builder/sections/"
    
    # Get all section files in order
    section_files = get_section_files_ordered(sections_dir)
    
    # Read each section
    complete_content = ""
    for section_file in section_files:
        section_content = read_file(section_file)
        complete_content += section_content + "\n\n"
    
    # Save assembled SKILL
    output_path = f"/home/claude/writer-friendly-skill-builder/{skill_name}-assembled.md"
    write_file(output_path, complete_content)
    
    print(f"✓ Complete SKILL assembled: {output_path}")
    
    return output_path
```

**What assembly includes:**
- All module content (in correct order)
- Section separators (clear divisions between modules)
- Complete instructions
- All examples
- All rules and constraints

**Assembly does NOT include:**
- YAML frontmatter (Module 3 handles this)
- .zip packaging (Module 3 handles this)

### File Structure During Building

While Module 2 is working, the directory looks like this:

```
/home/claude/writer-friendly-skill-builder/
├── SKILL.md (this file - Modules 1 & 2)
├── build-progress.txt (tracking file)
├── sections/
│   ├── module-1.txt (if multi-module)
│   ├── module-2.txt (if multi-module)
│   ├── module-3.txt (if multi-module)
│   └── main-module.txt (if single-module)
└── [skill-name]-assembled.md (after assembly complete)
```

### Error Handling

Things can go wrong. Module 2 handles common errors gracefully.

**Common errors and solutions:**

**Error:** skill-creator returns incomplete module
**Solution:** Tell user, save what exists, ask user if they want to retry or continue

**Error:** File save fails
**Solution:** Retry save, use alternative path, alert user

**Error:** Progress file corrupted
**Solution:** Rebuild progress from existing section files

**Error:** Token limit hit mid-section
**Solution:** Save partial work, provide resume instructions

**Error:** User interrupts process
**Solution:** All saved work persists, resume instructions provided

### What Module 2 Delivers

When Module 2 completes successfully, it provides:

1. **Complete SKILL content** - All sections written and assembled
2. **Saved section files** - Individual modules preserved
3. **Progress tracking** - Record of what was built
4. **Assembly file** - Complete SKILL ready for Module 3
5. **Token usage report** - How many tokens were used
6. **Confirmation message** - "SKILL build complete, ready for packaging"

**Deliverable format:**
```
SKILL BUILD COMPLETE ✓

Skill Name: [skill-name]
Type: [Single-module / Multi-module with N modules]
Total Sections: [N]
All Sections Saved: ✓
Assembly Complete: ✓

Files created:
- /home/claude/writer-friendly-skill-builder/sections/ [N files]
- /home/claude/writer-friendly-skill-builder/[skill-name]-assembled.md

Token Usage: [X] / 190,000 ([Y] remaining)

Status: Ready for Module 3 (YAML Packaging)
```

### Module 2 Complete Checklist

Before declaring Module 2 complete, verify:

- ✓ Specification from Module 1 was read correctly
- ✓ Workflow type determined (single or multi-module)
- ✓ All sections built through skill-creator
- ✓ Each section saved to individual file
- ✓ Progress tracker updated after each section
- ✓ Token usage monitored throughout
- ✓ All sections assembled into complete SKILL
- ✓ Assembly file created and saved
- ✓ User notified of completion
- ✓ Ready to pass to Module 3

### Usage Instructions for Module 2

**To use Module 2 after Module 1 completes:**

Module 1 will tell you: "Your specification is ready. Module 2 will build the SKILL incrementally."

Module 2 then automatically:
1. Takes the specification
2. Determines build strategy (single or multi-module)
3. Coordinates with skill-creator
4. Builds incrementally
5. Saves all work
6. Assembles complete SKILL
7. Reports completion

**You don't need to do anything.** Module 2 handles everything automatically and keeps you informed of progress.

**If interrupted:**
Just say: "Resume building [skill-name]" and Module 2 will pick up where it left off.

---

## Module 2 Complete

Module 2 (Incremental Builder) is now complete. This module handles:
- ✓ Section-by-section building strategy
- ✓ Auto-save functionality
- ✓ Progress tracking system
- ✓ Resume capability
- ✓ Token monitoring
- ✓ Coordination with skill-creator
- ✓ Assembly of complete SKILL
- ✓ Error handling

**Next:** Module 3 will add YAML frontmatter, validate everything, and package as .zip file ready for upload.

---

## Module 3: YAML Packager & Validator

This module takes the completed SKILL content from Module 2 and packages it properly with valid YAML frontmatter. It validates everything before packaging to prevent all 5 common upload errors.

### Purpose

Create error-free .zip packages ready for upload to Claude. Prevent ALL upload errors through comprehensive validation before packaging.

### The 5 Upload Errors This Module Prevents

Based on real user experience with 60+ SKILLs, these are the errors that MUST be caught:

**Error 1: Malformed YAML frontmatter**
- Problem: YAML syntax is invalid (wrong indentation, missing colons, etc.)
- Prevention: Parse and validate YAML before packaging
- Check: Test YAML syntax, verify proper structure

**Error 2: Unexpected key in SKILL.md frontmatter**
- Problem: YAML contains keys not in the approved list
- Prevention: Only use approved keys (name, description, license, allowed-tools, metadata)
- Check: Parse YAML and verify every key is valid

**Error 3: Skill name can only contain lowercase letters, numbers, and hyphens**
- Problem: Name has uppercase letters, spaces, or special characters
- Prevention: Module 1 formatted it, but double-check before packaging
- Check: Regex validation: ^[a-z0-9]+(-[a-z0-9]+)*$

**Error 4: SKILL.md must start with YAML frontmatter (---)**
- Problem: Missing opening delimiter or content before YAML
- Prevention: Always place --- at the very first line
- Check: First line of SKILL.md must be exactly "---"

**Error 5: Upload must end in .skill or .zip**
- Problem: File extension is wrong (.md instead of .zip)
- Prevention: Always package as .zip
- Check: Final filename must end with .zip

### Valid YAML Keys

**ONLY these keys are allowed:**

**Required:**
- `name` - Skill name (lowercase-with-hyphens-only)
- `description` - Clear description of what SKILL does and when to use it

**Optional:**
- `license` - License identifier (default: MIT)
- `allowed-tools` - Array of tool names (use empty array [] if none needed)
- `metadata` - Object with additional info (author, version, etc.)

**Any other key will cause upload error.**

### YAML Frontmatter Templates

**Basic template (most SKILLs):**
```yaml
---
name: skill-name-here
description: Clear description of what this skill does and when to use it
---
```

**With license (recommended):**
```yaml
---
name: skill-name-here
description: Clear description of what this skill does and when to use it
license: MIT
---
```

**With tools (if SKILL needs specific Claude tools):**
```yaml
---
name: skill-name-here
description: Clear description of what this skill does and when to use it
license: MIT
allowed-tools:
  - bash_tool
  - str_replace
  - create_file
---
```

**With metadata (optional additional info):**
```yaml
---
name: skill-name-here
description: Clear description of what this skill does and when to use it
license: MIT
metadata:
  author: Writer Name
  version: 1.0
  genre: romance
  created: 2025-11-14
---
```

### YAML Generation Process

**Step 1: Extract information from specification**

From Module 1's specification, extract:
- Skill name (already validated and formatted)
- Description (from Purpose section)
- Any tool requirements (from Rules section)

**Step 2: Determine which template to use**

- Basic needs: name + description only
- Standard: name + description + license (MIT)
- Tools needed: add allowed-tools array
- Extra info: add metadata object

Most SKILLs use the standard template (name + description + license).

**Step 3: Generate YAML frontmatter**

Create YAML with proper formatting:
- Start with `---` on its own line
- Add `name:` key with skill name
- Add `description:` key with description (use | for multi-line if needed)
- Add optional keys if needed
- End with `---` on its own line
- Add ONE blank line after closing `---`

**Example generation:**
```yaml
---
name: scene-expander
description: Takes brief scene descriptions (2-3 sentences) and expands them into full scenes with dialogue, action, and description. Outputs 800-1200 word scenes in third person limited POV, past tense. Never uses em dashes or ellipses.
license: MIT
---

```

### YAML Validation Checklist

Before packaging, run this complete validation:

**Syntax Validation:**
- [ ] YAML parses without errors
- [ ] Proper indentation (2 spaces for nested items)
- [ ] No tabs (spaces only)
- [ ] Colons followed by space
- [ ] Arrays properly formatted with `- item`
- [ ] Strings quoted if they contain special characters

**Structure Validation:**
- [ ] Starts with `---` on line 1
- [ ] No blank lines before first `---`
- [ ] Ends with `---` on its own line
- [ ] Exactly one blank line after closing `---`
- [ ] Content starts after the blank line

**Key Validation:**
- [ ] Only approved keys present (name, description, license, allowed-tools, metadata)
- [ ] No unexpected keys
- [ ] Required keys present (name, description)
- [ ] Optional keys properly formatted if present

**Value Validation:**
- [ ] `name` is lowercase-with-hyphens-only
- [ ] `name` has no leading or trailing hyphens
- [ ] `name` has no double hyphens (--)
- [ ] `description` is non-empty string
- [ ] `description` is clear and specific
- [ ] `license` is valid identifier (if present)
- [ ] `allowed-tools` is array (if present)
- [ ] `metadata` is object (if present)

### Name Validation Function

Double-check skill name before packaging (Module 1 should have formatted it, but verify):

```python
def validate_skill_name(name):
    """
    Validate skill name meets all requirements
    Returns: (is_valid, error_message)
    """
    import re
    
    # Check for valid characters only
    if not re.match(r'^[a-z0-9]+(-[a-z0-9]+)*$', name):
        return False, "Name can only contain lowercase letters, numbers, and hyphens"
    
    # Check doesn't start or end with hyphen
    if name.startswith('-') or name.endswith('-'):
        return False, "Name cannot start or end with hyphen"
    
    # Check for double hyphens
    if '--' in name:
        return False, "Name cannot contain double hyphens"
    
    # Check not too short or too long
    if len(name) < 3:
        return False, "Name must be at least 3 characters"
    if len(name) > 50:
        return False, "Name should be 50 characters or less"
    
    return True, "Valid"
```

### YAML Parsing Function

Parse YAML to verify it's valid:

```python
def validate_yaml_structure(yaml_text):
    """
    Parse YAML and validate structure
    Returns: (is_valid, parsed_data, error_message)
    """
    import yaml
    
    try:
        # Parse YAML
        data = yaml.safe_load(yaml_text)
        
        # Check required keys
        if 'name' not in data:
            return False, None, "Missing required key: name"
        if 'description' not in data:
            return False, None, "Missing required key: description"
        
        # Check for unexpected keys
        valid_keys = {'name', 'description', 'license', 'allowed-tools', 'metadata'}
        for key in data.keys():
            if key not in valid_keys:
                return False, None, f"Unexpected key: {key}"
        
        # Validate name format
        name_valid, name_error = validate_skill_name(data['name'])
        if not name_valid:
            return False, None, f"Invalid name: {name_error}"
        
        # All checks passed
        return True, data, "Valid"
        
    except yaml.YAMLError as e:
        return False, None, f"YAML syntax error: {str(e)}"
```

### File Structure Creation

Create the proper file structure for packaging:

**Directory structure:**
```
/home/claude/writer-friendly-skill-builder/package/
├── SKILL.md (YAML frontmatter + content from Module 2)
└── README.md (optional usage instructions)
```

**Creating SKILL.md:**
1. Take YAML frontmatter (generated and validated)
2. Add ONE blank line
3. Add complete SKILL content from Module 2
4. Save as SKILL.md

**Creating README.md (optional):**
Generate usage instructions:
```markdown
# [Skill Name]

[Description]

## How to Use

1. Upload this SKILL to your Claude Project
2. In any chat within that Project, type:

\`\`\`
Use my [skill-name] SKILL to [what it does]
\`\`\`

## Input

[What you provide to the SKILL]

## Output

[What the SKILL produces]

## Rules & Constraints

[Any specific rules from specification]

---

Built with Writer-Friendly SKILL Builder Wrapper
```

### Packaging Process

Create the .zip file with proper structure:

**Step 1: Create package directory**
```bash
mkdir -p /home/claude/writer-friendly-skill-builder/package
```

**Step 2: Generate SKILL.md**
```bash
# Combine YAML + content
cat > /home/claude/writer-friendly-skill-builder/package/SKILL.md << 'EOF'
---
name: skill-name
description: Description here
license: MIT
---

[Content from Module 2 assembly]
EOF
```

**Step 3: Generate README.md (optional)**
```bash
cat > /home/claude/writer-friendly-skill-builder/package/README.md << 'EOF'
[Usage instructions]
EOF
```

**Step 4: Create .zip file**
```bash
cd /home/claude/writer-friendly-skill-builder/package
zip -r ../skill-name.zip .
cd ..
```

**Step 5: Validate .zip structure**
```bash
# Check what's in the zip
unzip -l skill-name.zip

# Should show:
# SKILL.md
# README.md (if included)
```

**Step 6: Move to outputs**
```bash
mv /home/claude/writer-friendly-skill-builder/skill-name.zip /mnt/user-data/outputs/
```

### Complete Validation Report

After packaging, generate this report for the user:

```
═══════════════════════════════════════
SKILL PACKAGING COMPLETE ✓
═══════════════════════════════════════

Skill Name: [skill-name]
Package File: [skill-name].zip
Package Size: [X] KB

VALIDATION REPORT
═══════════════════════════════════════

YAML Frontmatter Validation:
  ✓ Starts with --- (line 1)
  ✓ Ends with --- (proper closing)
  ✓ No blank lines before opening ---
  ✓ One blank line after closing ---
  ✓ YAML syntax is valid
  ✓ Proper indentation (2 spaces)
  ✓ No tabs detected
  ✓ All strings properly formatted

Key Validation:
  ✓ Required keys present (name, description)
  ✓ Only approved keys used
  ✓ No unexpected keys detected
  Keys included: name, description, license[, allowed-tools][, metadata]

Skill Name Validation:
  ✓ Lowercase letters only
  ✓ Numbers allowed
  ✓ Hyphens only (no spaces or special chars)
  ✓ No leading/trailing hyphens
  ✓ No double hyphens
  ✓ Length: [N] characters (valid range)

Content Validation:
  ✓ SKILL content present
  ✓ Content follows YAML (not mixed)
  ✓ All modules included
  ✓ Proper formatting throughout

File Structure Validation:
  ✓ SKILL.md exists
  ✓ SKILL.md at root level of zip
  [✓ README.md included (optional)]
  ✓ Proper .zip structure

UPLOAD ERROR PREVENTION
═══════════════════════════════════════

All 5 Common Errors Prevented:
  ✓ Error 1: Malformed YAML - PREVENTED
  ✓ Error 2: Unexpected keys - PREVENTED
  ✓ Error 3: Invalid name format - PREVENTED
  ✓ Error 4: Missing --- delimiter - PREVENTED
  ✓ Error 5: Wrong file extension - PREVENTED

═══════════════════════════════════════
STATUS: READY FOR UPLOAD ✓
═══════════════════════════════════════

Download your SKILL:
[Link to computer:///mnt/user-data/outputs/skill-name.zip]

This SKILL has been thoroughly validated and is ready to upload to Claude without errors.

TO UPLOAD:
1. Download the .zip file using the link above
2. Go to Claude Settings → Skills
3. Click "Upload skill"
4. Select your [skill-name].zip file
5. Upload completes without errors ✓

TO USE:
In any Project where this SKILL is available, type:
"Use my [skill-name] SKILL to [what you want it to do]"
```

### Error Handling

If validation fails, provide clear error messages and fix suggestions:

**YAML Syntax Error:**
```
ERROR: YAML Validation Failed
━━━━━━━━━━━━━━━━━━━━━━━━━━

Problem: YAML syntax error on line [N]
Details: [specific error message]

Suggested Fix:
[Specific instructions based on error]

I can attempt to auto-fix this. Should I:
1. Auto-fix and continue packaging
2. Show you the YAML for manual review
3. Rebuild the YAML from scratch
```

**Invalid Key Error:**
```
ERROR: Unexpected Key in YAML
━━━━━━━━━━━━━━━━━━━━━━━━━━

Problem: YAML contains key '[key-name]' which is not approved

Approved keys: name, description, license, allowed-tools, metadata
Found unexpected: [key-name]

Fix: Removing '[key-name]' from YAML...

✓ Fixed. Continuing with packaging...
```

**Name Format Error:**
```
ERROR: Invalid Skill Name Format
━━━━━━━━━━━━━━━━━━━━━━━━━━

Problem: Skill name '[name]' contains invalid characters

Current name: [name]
Issue: [specific issue - uppercase/spaces/special chars]

Suggested fix: [corrected-name]

Apply this fix and continue? (yes/no)
```

### Workflow Integration

How Module 3 connects with Modules 1 & 2:

**Input from Module 2:**
- Complete SKILL content (all modules assembled)
- Skill name (validated by Module 1)
- Specification document (for description)

**Module 3 Process:**
1. Read assembled SKILL content
2. Extract skill name and description
3. Generate YAML frontmatter
4. Validate YAML completely
5. Create SKILL.md (YAML + content)
6. Optionally create README.md
7. Package as .zip
8. Run final validation
9. Move to outputs directory
10. Generate validation report
11. Provide download link

**Output to User:**
- Validated .zip file ready for upload
- Comprehensive validation report
- Usage instructions
- Confidence it will upload without errors

### Module 3 Complete Checklist

Before declaring Module 3 complete, verify:

- [ ] YAML frontmatter generated correctly
- [ ] All 5 upload errors validated against
- [ ] Validation functions implemented
- [ ] SKILL.md created properly (YAML + content)
- [ ] Optional README.md generation available
- [ ] Packaging process creates proper .zip
- [ ] .zip moved to /mnt/user-data/outputs/
- [ ] Validation report generated
- [ ] Download link provided
- [ ] Error handling comprehensive
- [ ] Clear user messaging throughout

### Usage Instructions for Module 3

**Module 3 runs automatically after Module 2 completes.**

Module 2 tells you: "SKILL build complete, ready for packaging."

Module 3 then automatically:
1. Takes the assembled SKILL content
2. Generates YAML frontmatter
3. Validates everything (runs all checks)
4. Creates SKILL.md file
5. Packages as .zip
6. Validates the final package
7. Provides download link with validation report

**You don't need to do anything.** Module 3 handles all packaging and validation automatically.

**If validation finds errors:**
Module 3 will attempt to auto-fix. If it can't fix automatically, it will show you the error and suggest fixes.

---

## Module 3 Complete

Module 3 (YAML Packager & Validator) is now complete. This module handles:
- ✓ YAML frontmatter generation
- ✓ Complete validation (all 5 upload errors)
- ✓ Syntax and structure checking
- ✓ Key and value validation
- ✓ File structure creation
- ✓ .zip packaging
- ✓ Validation report generation
- ✓ Error handling and auto-fixing
- ✓ Download link provision

---

## Complete Wrapper Summary

All three modules are now complete:

### Module 1: Discovery Interview
Asks writer-friendly questions and creates validated specifications.

### Module 2: Incremental Builder
Builds SKILLs section-by-section, saves work continuously, prevents token limit issues.

### Module 3: YAML Packager & Validator
Validates everything, prevents upload errors, packages as .zip, provides download link.

### Complete Workflow

**User Experience:**
1. User: "Use the writer-friendly-skill-builder to create a new SKILL"
2. Module 1: Asks 6 simple questions, validates answers, creates specification
3. Module 2: Coordinates with skill-creator, builds incrementally, saves all work
4. Module 3: Adds YAML, validates completely, packages as .zip
5. User: Downloads error-free .zip file, uploads to Claude successfully

**Writer Benefits:**
- ✓ No technical knowledge required
- ✓ No YAML formatting needed
- ✓ No token limit issues
- ✓ No upload errors
- ✓ Simple question-and-answer process
- ✓ Automatic validation and packaging
- ✓ Clear progress tracking
- ✓ Can resume if interrupted

**System Benefits:**
- ✓ Prevents all 5 common upload errors
- ✓ Handles token limits through incremental building
- ✓ Saves work at every checkpoint
- ✓ Validates before packaging (not after upload)
- ✓ Clear error messages and auto-fixing
- ✓ Comprehensive documentation

### Success Metrics Achieved

From the original blueprint, all success criteria met:

- ✓ Non-technical writer can answer simple questions
- ✓ Wrapper translates answers into proper skill-creator format
- ✓ SKILL builds in sections without hitting token limits
- ✓ YAML validation catches all 5 common errors
- ✓ Final .zip file uploads to Claude without errors
- ✓ Created SKILL works for its intended purpose

This wrapper is now complete and ready for use.
